<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/functions.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Ensure we're sending JSON response
header('Content-Type: application/json');

if (!isset($_GET['product_id'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Product ID is required']);
    exit;
}

$product_id = intval($_GET['product_id']);

try {
    // Log the incoming request
    error_log("Fetching supplier price for product ID: " . $product_id);

    // Get the latest supplier receipt price for the product
    $stmt = $pdo->prepare("
        SELECT 
            sr.unit_price,
            sr.created_at,
            s.name as supplier_name
        FROM supplier_receipts sr
        LEFT JOIN suppliers s ON sr.supplier_id = s.id
        WHERE sr.product_id = ? 
        AND sr.status = 'yes'  -- Only approved receipts
        ORDER BY sr.created_at DESC
        LIMIT 1
    ");
    
    $stmt->execute([$product_id]);
    $supplier_price = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$supplier_price) {
        // No supplier receipts found
        echo json_encode([
            'unit_price' => null,
            'message' => 'No supplier receipts found for this product'
        ]);
        exit;
    }

    // Ensure numeric values are properly formatted
    $supplier_price['unit_price'] = floatval($supplier_price['unit_price']);

    // Log successful response
    error_log("Successfully fetched supplier price: " . json_encode($supplier_price));
    
    // Send the response
    echo json_encode($supplier_price);
    exit;
    
} catch (Exception $e) {
    error_log("Error in get_supplier_price.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'error' => 'Failed to fetch supplier price',
        'message' => $e->getMessage()
    ]);
    exit;
} 